! Chebyshev Series Expansion and Interpolation routine
! subroutines
!     chebexp  : function f(x) -> Chebyshev series
!     chebeval : Chebyshev series -> evaluation of the f(x)
! necessary package
!     fft2f.f  : FFT package
!
!
! chebexp, chebeval
!     [description]
!         evaluation of f(x) from Chebyshev series of f(x)
!     [declaration]
!         real*8 chebeval
!         external f
!     [usage]
!         call chebexp(f, a, b, eps, lenc, c, err)  ! f(x) -> c()
!         ...
!         y = chebeval(x, c)  ! evaluation of the f(x) from c()
!     [parameters]
!         f         : function f(x) (real*8 function)
!         a         : lower limit of interpolation (real*8)
!         b         : upper limit of interpolation (real*8)
!         eps       : relative error of interpolation (real*8)
!         lenc      : (length of c()) - 1 (integer)
!         c         : data of Chebyshev expansion, 
!                     c(0...lenc) (real*8)
!         err       : estimate of the maximum absolute error 
!                     of the interpolation over [a,b] (real*8)
!     [remarks]
!         initial parameters
!             lenc >= 12
!             example :
!                 lenc = 1024 + 4
!         function
!             f(x) needs to be analytic over [a,b].
!         relative error
!             eps is relative error requested excluding 
!             cancellation of significant digits.
!             i.e. eps means : (maximum absolute error) / 
!                              (integral_a^b |f(x)| dx).
!             eps does not mean : (maximum absolute error) / I.
!         error message
!             err >= 0 : normal termination.
!             err < 0  : abnormal termination (n > lenc-4).
!                        i.e. convergent error is detected :
!                            1. f(x) or (d/dx)^n f(x) has 
!                               discontinuous points or sharp 
!                               peaks over [a,b].
!                            2. relative error of f(x) is 
!                               greater than eps.
!                            3. f(x) has oscillatory factor 
!                               and frequency of the oscillation 
!                               is very high.
!         array of c()
!             lenc           : int(c(0))
!             n              : int(c(1))
!             (b+a)/2        : c(2)
!             2/(b-a)        : c(3)
!             f(c(2)-t/c(3)) : c(lenc)*T_0(t) + c(lenc-1)*T_1(t) 
!                              + ... + c(lenc-n)*T_n(t)
!
!
      subroutine chebexp(f, a, b, eps, lenc, c, err)
      integer lenc
      real*8 f, a, b, eps, c(0 : lenc), err
      integer j, k, l, n
      real*8 esf, ba, cos2, sin2, wi, ss, x, y, t, h, eref, erefh, 
     &    errh
      esf = 10
      ba = 0.5d0 * (b - a)
      c(0) = 0.5d0 * f(a)
      c(2) = 0.5d0 * f(b)
      c(1) = f(a + ba)
      c(lenc - 1) = c(0) - c(2)
      c(lenc) = c(0) + c(2) + c(1)
      c(lenc - 2) = c(0) + c(2) - c(1)
      cos2 = 0
      sin2 = 1
      wi = -1
      h = 1
      l = 1
      n = 2
   10 continue
          ss = 2 * sin2
          cos2 = sqrt(2 + cos2)
          sin2 = sin2 / (2 + cos2)
          x = ba * sin2
          y = 0
          do j = 0, l - 1
              x = x + y
              y = y + ss * (ba - x)
              c(j) = f(a + x)
              c(n - 1 - j) = f(b - x)
          end do
          wi = wi / cos2
          call ddct(n, 0.5d0 * cos2, wi, c)
          l = n
          n = n * 2
          do j = l - 1, 0, -1
              k = lenc - j
              t = c(k) - c(j)
              c(k) = c(k) + c(j)
              c(lenc - n + j) = t
          end do
          if (n .eq. 4) then
              eref = 0.25d0 * (abs(c(lenc)) + abs(c(lenc - 1)) + 
     &            abs(c(lenc - 2)) + abs(c(lenc - 3)) + 
     &            abs(c(lenc - 4)))
              erefh = eref * sqrt(eps)
              eref = eref * eps
              err = erefh
          end if
          h = h * 0.5d0
          errh = esf * err
          err = h * (0.5d0 * abs(c(lenc - n)) + abs(c(lenc - n + 1)))
      if ((err .gt. eref .or. errh .gt. erefh) .and. 
     &    2 * n + 4 .le. lenc) goto 10
      c(lenc - n) = c(lenc - n) * 0.5d0
      c(lenc) = c(lenc) * 0.5d0
      do j = lenc - n, lenc
          c(j) = c(j) * h
      end do
      if (err .gt. eref .or. errh .gt. erefh) then
          err = -err
      else
   20     continue
              n = n - 2
              err = err + (abs(c(lenc - n)) + 
     &            abs(c(lenc - n + 1)))
          if (err .lt. eref .and. n .gt. 2) goto 20
          n = n + 2
          err = eref
      end if
      if (ba .ne. 0) then
          c(3) = 1 / ba
      else
          c(3) = 0
      end if
      c(2) = 0.5d0 * (b + a)
      c(1) = n + 0.5d0
      c(0) = lenc + 0.5d0
      end
!
      real*8 function chebeval(x, c)
      real*8 c(0 : *), x
      integer lenc, n, j
      real*8 t, t2, v0, v1
      lenc = int(c(0))
      n = int(c(1))
      t = (c(2) - x) * c(3)
      t2 = 2 * t
      v0 = c(lenc - n)
      v1 = c(lenc - n + 1) + t2 * v0
      do j = lenc - n + 2, lenc - 2, 2
          v0 = c(j) + t2 * v1 - v0
          v1 = c(j + 1) + t2 * v0 - v1
      end do
      chebeval = c(lenc) + t * v1 - v0
      end
!
