/*
Clenshaw-Curtis-Quadrature
Numerical Automatic Integrator
    method    : Chebyshev Series Expansion
    dimension : one
    table     : not use
function
    intcc  : integrator of f(x) over [a,b].
necessary package
    fft2f.c  : FFT package
*/

/*
intcc
    [description]
        I = integral of f(x) over [a,b]
    [declaration]
        void intcc(double (*f)(double), double a, double b, double eps, 
            int lenc, double *c, double *i, double *err);
    [usage]
        intcc(f, a, b, eps, lenc, c, &i, &err);
    [parameters]
        f         : integrand f(x) (double (*f)(double))
        a         : lower limit of integration (double)
        b         : upper limit of integration (double)
        eps       : relative error requested (double)
        lenc      : (length of c[]) - 1 (int)
        c         : work area for Chebyshev expansion, 
                    c[0...lenc] (double *)
        i         : approximation to the integral (double *)
        err       : estimate of the absolute error (double *)
    [remarks]
        initial parameters
            lenc >= 8 and 
            lenc > (maximum number of f(x) evaluations) / 2
            example :
                lenc = 1024;
        function
            f(x) needs to be analytic over [a,b].
        relative error
            eps is relative error requested excluding 
            cancellation of significant digits.
            i.e. eps means : (absolute error) / 
                             (integral_a^b |f(x)| dx).
            eps does not mean : (absolute error) / I.
        error message
            err >= 0 : normal termination.
            err < 0  : abnormal termination (n > nmax).
                       i.e. convergent error is detected :
                           1. f(x) or (d/dx)^n f(x) has 
                              discontinuous points or sharp 
                              peaks over [a,b].
                              you must use other routine.
                           2. relative error of f(x) is 
                              greater than eps.
                           3. f(x) has oscillatory factor 
                              and frequency of the oscillation 
                              is very high.
*/


#include <math.h>

void intcc(double (*f)(double), double a, double b, double eps, 
    int lenc, double *c, double *i, double *err)
{
    void ddct(int, double, double, double *);
    int j, k, l, n;
    double eref, t0, t1, ba, cos2, sin2, wi, ss, x, y;

    ba = 0.5 * (b - a);
    cos2 = 2 + sqrt(2.0);
    sin2 = 1 / cos2;
    x = ba * sin2;
    c[0] = 0.5 * (*f)(a);
    c[1] = 0.5 * (*f)(b);
    c[2] = (*f)(a + ba);
    c[3] = (*f)(a + x);
    c[4] = (*f)(b - x);
    eref = eps * (fabs(c[0]) + fabs(c[1]) + fabs(c[2]) + fabs(c[3]) + 
        fabs(c[4]));
    t0 = c[0] + c[1];
    t1 = c[3] + c[4];
    c[lenc - 1] = t0 - c[2];
    t0 += c[2];
    c[lenc - 2] = t0 - t1;
    c[lenc] = t0 + t1;
    wi = -1;
    l = 2;
    n = 4;
    do {
        ss = 2 * sin2;
        cos2 = 2 + sqrt(cos2);
        sin2 /= cos2;
        x = ba * sin2;
        y = 0;
        for (j = 0; j <= l - 1; j++) {
            x += y;
            y += ss * (ba - x);
            c[j] = (*f)(a + x) + (*f)(b - x);
        }
        wi /= 2 - ss;
        ddct(l, 0.5 * (2 - ss), wi, c);
        for (j = l - 1; j >= 0; j--) {
            k = lenc - j;
            t1 = c[k] - c[j];
            c[k] += c[j];
            c[lenc - n + j] = t1;
        }
        *err = fabs(c[lenc - n]) + fabs(c[lenc - n + 1]);
        eref *= 2;
        l = n;
        n *= 2;
    } while (*err > eref * n && n <= lenc);
    c[lenc] *= 0.5;
    c[lenc - l] *= 0.5;
    *i = 0;
    for (j = l; j >= 0; j--) {
        *i += c[lenc - j] / (1 - 4 * j * j);
    }
    *i *= (b - a) / l;
    if (*err > eref * n) {
        *err *= -fabs((b - a) / l);
    } else {
        *err = eref * fabs((b - a) / l);
    }
}

