! Clenshaw-Curtis-Quadrature
! Numerical Automatic Integrator
!     method    : Chebyshev Series Expansion
!     dimension : one
!     table     : use
! subroutine
!     intcc  : integrator of f(x) over [a,b].
! necessary package
!     fft2f.f  : FFT package
!
!
! intcc
!     [description]
!         I = integral of f(x) over [a,b]
!     [declaration]
!         external f
!     [usage]
!         call intccini(lenw, w)  ! initialization of w
!         ...
!         call intcc(f, a, b, eps, lenw, w, i, err)
!     [parameters]
!         lenw      : (length of w()) - 1 (integer)
!         w         : work area and weights of the quadrature 
!                     formula, w(0...lenw) (real*8)
!         f         : integrand f(x) (real*8 function)
!         a         : lower limit of integration (real*8)
!         b         : upper limit of integration (real*8)
!         eps       : relative error requested (real*8)
!         i         : approximation to the integral (real*8)
!         err       : estimate of the absolute error (real*8)
!     [remarks]
!         initial parameters
!             lenw >= 14 and 
!             lenw > (maximum number of f(x) evaluations) * 3 / 2
!             example :
!                 lenc = 3200
!         function
!             f(x) needs to be analytic over [a,b].
!         relative error
!             eps is relative error requested excluding 
!             cancellation of significant digits.
!             i.e. eps means : (absolute error) / 
!                              (integral_a^b |f(x)| dx).
!             eps does not mean : (absolute error) / I.
!         error message
!             err >= 0 : normal termination.
!             err < 0  : abnormal termination (n > nmax).
!                        i.e. convergent error is detected :
!                            1. f(x) or (d/dx)^n f(x) has 
!                               discontinuous points or sharp 
!                               peaks over [a,b].
!                               you must use other routine.
!                            2. relative error of f(x) is 
!                               greater than eps.
!                            3. f(x) has oscillatory factor 
!                               and frequency of the oscillation 
!                               is very high.
!
!
      subroutine intccini(lenw, w)
      integer lenw
      real*8 w(0 : lenw)
      integer j, k, l, m
      real*8 cos2, sin1, sin2, hl, wi
      cos2 = 0
      sin1 = 1
      sin2 = 1
      hl = 0.5d0
      k = lenw
      l = 2
      do while (l .lt. k - l - 1)
          w(0) = hl * 0.5d0
          do j = 1, l
              w(j) = hl / (1 - 4 * j * j)
          end do
          w(l) = w(l) * 0.5d0
          wi = sin1
          call dfct(l, 0.5d0 * cos2, wi, w)
          cos2 = sqrt(2 + cos2)
          sin1 = sin1 / cos2
          sin2 = sin2 / (2 + cos2)
          w(k) = sin2
          w(k - 1) = w(0)
          w(k - 2) = w(l)
          k = k - 3
          m = l
          do while (m .gt. 1)
              m = m / 2
              do j = m, l - m, 2 * m
                  w(k) = w(j)
                  k = k - 1
              end do
          end do
          hl = hl * 0.5d0
          l = l * 2
      end do
      end
!
      subroutine intcc(f, a, b, eps, lenw, w, i, err)
      integer lenw
      real*8 w(0 : lenw), f, a, b, i, eps, err
      integer j, k, l
      real*8 esf, eref, erefh, hh, ir, iback, irback, ba, ss, x, 
     &    y, fx, errir
      esf = 10
      ba = 0.5d0 * (b - a)
      ss = 2 * w(lenw)
      x = ba * w(lenw)
      w(0) = 0.5d0 * f(a)
      w(3) = 0.5d0 * f(b)
      w(2) = f(a + x)
      w(4) = f(b - x)
      w(1) = f(a + ba)
      eref = 0.5d0 * (abs(w(0)) + abs(w(1)) + abs(w(2)) + 
     &    abs(w(3)) + abs(w(4)))
      w(0) = w(0) + w(3)
      w(2) = w(2) + w(4)
      ir = w(0) + w(1) + w(2)
      i = w(0) * w(lenw - 1) + w(1) * w(lenw - 2) + w(2) * 
     &    w(lenw - 3)
      erefh = eref * sqrt(eps)
      eref = eref * eps
      hh = 0.25d0
      l = 2
      k = lenw - 5
   10 continue
          iback = i
          irback = ir
          x = ba * w(k + 1)
          y = 0
          i = w(0) * w(k)
          do j = 1, l
              x = x + y
              y = y + ss * (ba - x)
              fx = f(a + x) + f(b - x)
              ir = ir + fx
              i = i + (w(j) * w(k - j) + fx * w(k - j - l))
              w(j + l) = fx
          end do
          ss = 2 * w(k + 1)
          err = esf * l * abs(i - iback)
          hh = hh * 0.25d0
          errir = hh * abs(ir - 2 * irback)
          l = l * 2
          k = k - (l + 2)
      if ((err .gt. erefh .or. errir .gt. eref) .and. 
     &    k .gt. 4 * l) goto 10
      i = i * (b - a)
      if (err .gt. erefh .or. errir .gt. eref) then
          err = -err * abs(b - a)
      else
          err = eref * abs(b - a)
      end if
      end
!
